import sys
import os
import time
import difflib
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.keys import Keys
from selenium.common.exceptions import (TimeoutException, StaleElementReferenceException,
                                        ElementClickInterceptedException)

# 导入图片处理类和对话类（与你的代码完全一致）
from lycphotochange import ImageProcessor
from dblyc import DoubaoTextChat
from dslyc import DeepseekTextChat


# 新增：资源路径适配函数（解决打包后路径问题，不影响原有操作步骤）
def get_resource_path(relative_path):
    """适配PyInstaller打包后的资源路径：
    - 未打包时使用当前目录
    - 打包后使用临时解压目录sys._MEIPASS
    """
    if getattr(sys, 'frozen', False):
        return os.path.join(sys._MEIPASS, relative_path)
    else:
        return os.path.join(os.getcwd(), relative_path)


def should_press_esc(lyc1):
    """判断是否需要按ESC（最高优先级）"""
    if "班" in lyc1 or "劳动" in lyc1 or "国旗" in lyc1:
        if "国旗班" in lyc1 or "班班有歌声" in lyc1:
            print(f"⚠️ 文本含特殊关键词→按ESC")
            return True
        else:
            print(f"🔍 文本含常规关键词→继续操作")
            return False
    else:
        print(f"⚠️ 文本不含关键词→按ESC")
        return True


def get_matching_file(lyc1, folder_path):
    """智能匹配文件（第二优先级）"""
    print(f"\n===== 开始文件匹配（当前文本：{lyc1[:30]}...）=====")
    # 国旗场景：必须完整包含“国旗”且不含“班”字
    if "国旗" in lyc1 and "班" not in lyc1:
        flag_files = [f for f in os.listdir(folder_path)
                      if '国旗' in f and f.lower().endswith(('.jpg', '.jpeg', '.png'))]
        if flag_files:
            matched_file = flag_files[0]
            file_path = os.path.join(folder_path, matched_file)
            print(f"🔍 特殊匹配（国旗）：文本含“国旗”且不含“班”→选中 {matched_file}")
            return file_path
        else:
            print("⚠️ 文本含“国旗”且不含“班”但未找到对应图片，回退到智能匹配")
    # 劳动场景：必须完整包含“劳动”且不含“班”字
    if "劳动" in lyc1 and "班" not in lyc1:
        labor_files = [f for f in os.listdir(folder_path)
                       if '劳动' in f and f.lower().endswith(('.jpg', '.jpeg', '.png'))]
        if labor_files:
            matched_file = labor_files[0]
            file_path = os.path.join(folder_path, matched_file)
            print(f"🔍 特殊匹配（劳动）：文本含“劳动”且不含“班”→选中 {matched_file}")
            return file_path
        else:
            print("⚠️ 文本含“劳动”且不含“班”但未找到对应图片，回退到智能匹配")
    # 常规智能匹配
    target_text = lyc1.strip()
    files = [f for f in os.listdir(folder_path) if f.lower().endswith(('.jpg', '.jpeg', '.png'))]
    if not files:
        print("❌ 文件夹中无图片文件")
        return None
    file_names = [os.path.splitext(f)[0] for f in files]
    best_matches = difflib.get_close_matches(target_text, file_names, n=1, cutoff=0.3)
    if best_matches:
        best_file = [f for f in files if os.path.splitext(f)[0] == best_matches[0]][0]
        print(f"🔍 智能匹配：文本不含“国旗”/“劳动”或含“班”→{target_text[:30]}... → {best_file}")
        return os.path.join(folder_path, best_file)
    print(f"⚠️ 无匹配文件，使用第一个文件：{files[0]}")
    return os.path.join(folder_path, files[0])


def fill_form(driver, step_index, lyc1):
    """填充表单（第三优先级）"""
    print(f"===== 填充表单（步骤 {step_index}）=====")
    # 活动名称和地点逻辑（与你的代码完全一致）
    activity_name = ""
    activity_location = ""
    if "班" in lyc1:
        activity_name = "班会"
        activity_location = "教室内"
    elif "劳动" in lyc1:
        activity_name = "劳动实践"
        activity_location = "学校的草地上 "
    elif "国旗" in lyc1:
        activity_name = "升国旗仪式"
        activity_location = "学校操场"
    # 填充活动名称
    try:
        name_input = WebDriverWait(driver, 3).until(
            EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[5]/div[1]/input'))
        )
        name_input.clear()
        name_input.send_keys(activity_name or "校园活动")
        print(f"✅ 活动名称：{activity_name or '校园活动'}")
    except Exception as e:
        print(f"❌ 活动名称输入失败：{e}")
    # 填充活动地点
    if activity_location:
        try:
            location_input = WebDriverWait(driver, 3).until(
                EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[5]/div[2]/input'))
            )
            location_input.clear()
            location_input.send_keys(activity_location)
            print(f"✅ 活动地点：{activity_location}")
        except Exception as e:
            print(f"❌ 活动地点输入失败：{e}")
    # 填充时长
    try:
        duration_input = WebDriverWait(driver, 3).until(
            EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[5]/div[3]/input'))
        )
        duration_input.clear()
        duration_input.send_keys("1")
        print(f"✅ 活动时长：1小时")
    except Exception as e:
        print(f"❌ 时长输入失败：{e}")
    # 点击目标元素
    try:
        target_element = WebDriverWait(driver, 5).until(
            EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[5]/div[4]/label[1]/span[2]'))
        )
        target_element.click()
        print(f"✅ 点击目标元素")
    except Exception as e:
        print(f"⚠️ 目标元素点击失败：{e}")
    # 处理下拉框
    try:
        dropdown1 = WebDriverWait(driver, 3).until(
            EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[5]/div[4]/div/div/input'))
        )
        dropdown1.click()
        time.sleep(0.5)
        option1 = WebDriverWait(driver, 3).until(
            EC.element_to_be_clickable((By.XPATH, '/html/body/div[3]/div[1]/div[1]/ul/li[1]/span'))
        )
        option1.click()
        print(f"✅ 下拉框1：选择第1项")
    except Exception as e:
        print(f"❌ 下拉框1操作失败：{e}")
    try:
        dropdown2 = WebDriverWait(driver, 3).until(
            EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[5]/div[4]/div/div[1]/input'))
        )
        dropdown2.click()
        time.sleep(0.5)
        option5 = WebDriverWait(driver, 3).until(
            EC.element_to_be_clickable((By.XPATH, '/html/body/div[3]/div[1]/div[1]/ul/li[5]/span'))
        )
        option5.click()
        print(f"✅ 下拉框2：选择第5项")
    except Exception as e:
        print(f"❌ 下拉框2操作失败：{e}")


def upload_file(driver, lyc1, folder_path):
    """文件上传（最低优先级）"""
    print(f"===== 开始文件上传（文本：{lyc1[:30]}...）=====")
    file_path = get_matching_file(lyc1, folder_path)
    if not file_path:
        print("❌ 无匹配文件，跳过上传")
        return False
    try:
        file_input = WebDriverWait(driver, 10).until(
            EC.presence_of_element_located((By.XPATH, '//*[@id="pane-0"]/div/div[7]/div/div/input'))
        )
        file_input.send_keys(file_path)
        print(f"✅ 上传文件：{os.path.basename(file_path)}")
        try:
            WebDriverWait(driver, 5).until(
                EC.presence_of_element_located((By.XPATH, '//img[contains(@src, "upload")]'))
            )
            print("✅ 上传成功（预览图可见）")
            return True
        except TimeoutException:
            print("⚠️ 未检测到预览图，但文件可能已上传")
            return True
    except Exception as e:
        print(f"❌ 文件上传失败：{e}")
        return False


def click_final_button(driver, step_index):
    """点击步骤完成后的按钮"""
    print(f"===== 点击完成按钮（步骤 {step_index}）=====")
    try:
        button = WebDriverWait(driver, 10).until(
            EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[7]/button/span'))
        )
        button.click()
        print(f"✅ 已点击完成按钮")
        return True
    except Exception as e:
        print(f"❌ 点击完成按钮失败：{e}")
        return False


def main():
    # 修正：使用资源路径函数获取lycnb文件夹路径（解决打包后找不到文件夹的问题）
    folder_path = get_resource_path('lycnb')

    # ===== 处理lycnb文件夹中的图片（与你的代码逻辑完全一致）=====
    print("\n===== 开始处理lycnb文件夹中的图片 =====")
    try:
        image_processor = ImageProcessor(folder_path)
        image_processor.process_images()
        print("✅ 图片处理完成")
    except Exception as e:
        print(f"❌ 图片处理出错：{e}")
    # ==================================================

    # 检查并创建lycnb文件夹（与你的代码完全一致）
    if not os.path.exists(folder_path):
        try:
            os.makedirs(folder_path)
            print(f"\n✅ 已创建文件夹：{folder_path}")
            print("📌 请放入图片（含“国旗”或“劳动”的图片优先命名为含对应关键词）")
            input("按Enter键关闭程序（放入图片后重启）...")
            return
        except Exception as e:
            print(f"❌ 创建文件夹失败：{e}")
            return

    # 检查lycnb文件夹是否为空（与你的代码完全一致）
    if not os.listdir(folder_path):
        print(f"\n❌ 文件夹 {folder_path} 为空！")
        print("📌 请放入图片后重启程序")
        input("按Enter键关闭程序...")
        return

    # 询问是否跳过步骤（与你的代码完全一致）
    print("\n是否需要跳过前N个步骤？（需要请输入数字，不需要直接按回车）")
    skip_input = input().strip()
    skip_count = int(skip_input) if skip_input.isdigit() else 0
    print(f"🔍 已设置：跳过前 {skip_count} 个步骤\n")

    # 选择使用的对话类（与你的代码完全一致）
    print("请选择使用的类：")
    print("1. DoubaoTextChat")
    print("2. DeepseekTextChat")
    choice = input("输入数字选择（1或2）：")
    while choice not in ['1', '2']:
        print("输入无效，请输入1或2。")
        choice = input("输入数字选择（1或2）：")
    choice = int(choice)

    # 初始化浏览器（修正：使用资源路径函数获取chromedriver路径）
    chrome_options = Options()
    chrome_options.add_argument("--disable-notifications")
    chrome_options.add_experimental_option("excludeSwitches", ["enable-automation"])
    chrome_options.add_argument("--disable-blink-features=AutomationControlled")
    # 关键修复：打包后能找到chromedriver.exe
    driver_path = get_resource_path('chromedriver.exe')
    driver = webdriver.Chrome(service=Service(driver_path), options=chrome_options)

    try:
        # 打开对应聊天页面（与你的代码完全一致）
        if choice == 1:
            driver.get("https://doubao.com")
            print("✅ 已打开豆包页面")
        else:
            driver.get("https://chat.deepseek.com/sign_in")
            print("✅ 已打开 https://chat.deepseek.com/sign_in 页面")

        # 打开业务页面（与你的代码完全一致）
        driver.execute_script("window.open('');")
        driver.switch_to.window(driver.window_handles[1])
        driver.get("https://nazhisoft.com")
        print("✅ 已打开业务页面")

        # 等待用户登录（与你的代码完全一致）
        print("\n请在两个页面完成登录后按Enter键继续...")
        input()

        # 初始化聊天类（与你的代码完全一致）
        if choice == 1:
            driver.switch_to.window(driver.window_handles[0])
            doubao = DoubaoTextChat(driver_path=driver_path)  # 使用修正后的driver_path
            doubao.driver = driver
            print("===== 初始化豆包 =====")
            doubao.lyc666("你好")
            time.sleep(5)
            print("✅ 豆包初始化完成")
        else:
            driver.switch_to.window(driver.window_handles[0])
            deepseek = DeepseekTextChat(driver_path=driver_path)  # 使用修正后的driver_path
            deepseek.driver = driver
            WebDriverWait(driver, 30).until(
                EC.presence_of_element_located((By.XPATH, '//*[@id="chat-input"]'))
            )
            print("检测到登录成功，已准备好进行文字对话")

        # 生成自我评价（与你的代码完全一致）
        print("\n===== 生成自我评价 =====")
        task = "首先全程不要开启AI写作助手，写一个600字的高中生自我评价"
        formatted_task = f"{task}。【格式要求】严格控制字数，不加任何多余内容，不要分段，输入前务必清空内容"
        if choice == 1:
            response = doubao.lyc666(formatted_task)
        else:
            response = deepseek.send_message(formatted_task)

        if not response:
            print("❌ 获取回复失败，程序终止")
            return
        print(f"✅ 已获取回复，字数：{len(response)}")

        # 填充业务页面（与你的代码完全一致）
        driver.switch_to.window(driver.window_handles[1])
        try:
            textarea = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable(
                    (By.XPATH, '//*[@id="app"]/section/main/div/div/div[2]/div[1]/div/div/div/div[2]/div/textarea'))
            )
            textarea.click()
            textarea.send_keys(Keys.CONTROL + "a")
            textarea.send_keys(Keys.DELETE)
            textarea.send_keys(response)
            print("✅ 内容已填充到业务页面")
            submit_btn = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable(
                    (By.XPATH, '//*[@id="app"]/section/main/div/div/div[2]/div[1]/div/div/div/div[3]/button'))
            )
            submit_btn.click()
            print("✅ 已点击提交按钮")
        except Exception as e:
            print(f"❌ 填充内容或提交失败：{e}")

        # 多步骤处理（与你的代码完全一致）
        print("\n===== 开始多步骤处理 =====")
        try:
            button = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable((By.XPATH, '//*[@id="app"]/section/header/div/div/div/div[2]/ul/li[2]/a'))
            )
            button.click()
            print("✅ 已导航到目标页面")
            tab = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable((By.XPATH, '//*[@id="tab-0"]'))
            )
            tab.click()
            print("✅ 已切换到标签页0")

            max_steps = 100
            step_index = 1
            while step_index <= max_steps:
                if step_index <= skip_count:
                    print(f"⏩ 跳过步骤 {step_index}（用户指定）")
                    step_index += 1
                    continue

                print(f"\n===== 处理步骤 {step_index} =====")
                step_xpath = f'//*[@id="pane-0"]/div/ul/li[{step_index}]'
                button_xpath = f'//*[@id="pane-0"]/div/ul/li[{step_index}]/em/b'

                try:
                    step_element = WebDriverWait(driver, 3).until(
                        EC.presence_of_element_located((By.XPATH, step_xpath))
                    )
                    button = WebDriverWait(driver, 5).until(
                        EC.element_to_be_clickable((By.XPATH, button_xpath))
                    )
                    button.click()
                    print(f"✅ 步骤按钮点击成功")
                    time.sleep(1.5)

                    # 提取文本
                    lyc1 = driver.execute_script("""
                        var xpath = '/html/body/div[1]/section/main/div/div/div[2]/div[2]/div[3]/div/div[2]/div/div/div/div[2]/div/div/div[3]/text()';
                        var result = document.evaluate(xpath, document, null, XPathResult.STRING_TYPE, null);
                        return result.stringValue.trim() || '【文本为空】';
                    """)
                    print(f"✅ 文本提取成功：{lyc1[:50]}...（{len(lyc1)}字）")

                    # 判断是否按ESC
                    if should_press_esc(lyc1):
                        driver.find_element(By.TAG_NAME, 'body').send_keys(Keys.ESCAPE)
                        time.sleep(0.5)
                        driver.find_element(By.TAG_NAME, 'body').send_keys(Keys.ESCAPE)
                        print(f"✅ 已按ESC关闭弹窗")
                        step_index += 1
                        continue

                    # 匹配文件
                    file_to_upload = get_matching_file(lyc1, folder_path)
                    if file_to_upload:
                        print(f"🔍 已确定上传文件：{os.path.basename(file_to_upload)}")
                    else:
                        print("⚠️ 未找到匹配文件，使用默认上传逻辑")

                    # 填充表单
                    fill_form(driver, step_index, lyc1)

                    # 处理文本区域
                    try:
                        textarea_element = WebDriverWait(driver, 2).until(
                            EC.presence_of_element_located((By.XPATH, '//*[@id="pane-0"]/div/div[6]/textarea'))
                        )
                        if "劳动" in lyc1 and "班" not in lyc1:
                            lyc_task = (f"请帮我写一段男高中生在学校的大草地上进行{lyc1}的写实，200字左右"
                                        "。【格式要求】严格控制字数，不加多余内容，不要分段，输入前清空内容")
                            print(f"🔍 使用劳动场景提示词：{lyc_task[:50]}...")
                        else:
                            lyc_task = f"请帮我书写一份关于{lyc1}的写实描述，150字左右。【格式要求】严格控制字数，不加多余内容，不要分段，输入前清空内容"
                            print(f"🔍 使用常规提示词：{lyc_task[:50]}...")

                        # 切换到聊天窗口发送请求
                        driver.switch_to.window(driver.window_handles[0])
                        if choice == 1:
                            lyc_response = doubao.lyc666(lyc_task)
                        else:
                            lyc_response = deepseek.send_message(lyc_task)

                        # 将回复填入业务页面
                        if lyc_response:
                            driver.switch_to.window(driver.window_handles[1])
                            textarea_element = WebDriverWait(driver, 10).until(
                                EC.element_to_be_clickable((By.XPATH, '//*[@id="pane-0"]/div/div[6]/textarea'))
                            )
                            textarea_element.send_keys(lyc_response)
                            print(f"✅ 回复内容已填入")
                    except Exception as e:
                        print(f"❌ 文本区域操作失败：{e}")

                    # 上传文件
                    upload_success = upload_file(driver, lyc1, folder_path)
                    if upload_success:
                        if click_final_button(driver, step_index):
                            driver.find_element(By.TAG_NAME, 'body').send_keys(Keys.ESCAPE)
                            time.sleep(0.5)
                            driver.find_element(By.TAG_NAME, 'body').send_keys(Keys.ESCAPE)
                            print("✅ 已关闭弹窗")

                    step_index += 1

                except TimeoutException:
                    print(f"❌ 未找到步骤 {step_index} 的元素，处理结束")
                    break
                except Exception as e:
                    print(f"❌ 步骤 {step_index} 出错：{e}")
                    step_index += 1
                    continue

            print(f"\n===== 处理完成 =====")
            print(f"共处理 {step_index - 1} 个步骤（跳过前 {skip_count} 步）")

        except Exception as e:
            print(f"❌ 执行额外任务出错：{e}")

        input("\n任务完成，按Enter键关闭程序...")

    except Exception as e:
        print(f"❌ 程序运行出错：{e}")

    finally:
        # 关闭浏览器（与你的代码完全一致）
        if 'doubao' in locals() and doubao.driver:
            doubao.close()
        elif 'deepseek' in locals() and deepseek.driver:
            deepseek.close()
        driver.quit()


if __name__ == "__main__":
    # 修正：使用资源路径函数检查chromedriver是否存在
    if not os.path.exists(get_resource_path("chromedriver.exe")):
        print("❌ 错误：未找到chromedriver.exe")
        sys.exit(1)
    main()