import os
import tempfile
from PIL import Image  # 需确保已安装：pip install pillow


class ImageProcessor:
    """图片处理器：将非JPG格式转为JPG，压缩大于0.5MB的图片"""

    def __init__(self, source_dir):
        """初始化图片处理器
        Args:
            source_dir: 源图片文件夹路径（lycnb文件夹）
        """
        self.source_dir = source_dir  # 接收外部传入的文件夹路径

    def process_images(self):
        """处理文件夹中所有图片（入口方法，供main.py调用）"""
        if not os.path.exists(self.source_dir):
            print(f"⚠️ 图片文件夹不存在：{self.source_dir}")
            return

        # 遍历文件夹中所有文件
        for filename in os.listdir(self.source_dir):
            source_path = os.path.join(self.source_dir, filename)

            # 跳过子文件夹，只处理文件
            if not os.path.isfile(source_path):
                continue

            try:
                # 处理单个图片（转换格式+压缩）
                self._process_single_image(filename, source_path)
            except Exception as e:
                print(f"❌ 处理图片 {filename} 失败: {e}")

    def _process_single_image(self, filename, source_path):
        """处理单个图片（私有方法，内部调用）"""
        # 分离文件名和扩展名（如"image.png" → "image" + ".png"）
        base_name, ext = os.path.splitext(filename)
        target_ext = '.jpg'  # 目标格式：JPG

        # 获取图片大小（单位：MB）
        file_size = os.path.getsize(source_path) / (1024 * 1024)

        # 步骤1：非JPG格式转为JPG
        if ext.lower() != target_ext:
            print(f"🔄 转换 {filename} 为JPG格式")
            # 调用转换方法，获取临时文件路径
            temp_file = self._convert_to_jpg(source_path)

            # 删除原图，将临时文件重命名为原图名（替换为JPG）
            os.remove(source_path)
            new_filename = f"{base_name}{target_ext}"
            new_path = os.path.join(self.source_dir, new_filename)
            os.rename(temp_file, new_path)

            # 更新路径（后续压缩使用新路径）
            source_path = new_path
            filename = new_filename

        # 步骤2：大于0.5MB的图片压缩
        if file_size > 0.5:
            print(f"📉 压缩 {filename}（原始大小：{file_size:.2f} MB）")
            # 调用压缩方法，获取临时文件路径
            temp_file = self._compress_image(source_path)

            # 删除原图，用压缩后的临时文件替换
            os.remove(source_path)
            os.rename(temp_file, source_path)

            # 输出压缩后大小
            compressed_size = os.path.getsize(source_path) / (1024 * 1024)
            print(f"✅ 压缩完成（大小：{compressed_size:.2f} MB）")

    def _convert_to_jpg(self, source_path):
        """将图片转换为JPG格式（处理PNG等其他格式）"""
        try:
            # 创建临时文件（避免覆盖原图）
            temp_file = tempfile.mktemp(suffix='.jpg')

            # 打开图片并转换为RGB模式（JPG不支持透明通道）
            with Image.open(source_path) as img:
                if img.mode in ('RGBA', 'P', 'L'):  # 处理透明/灰度图
                    img = img.convert('RGB')
                img.save(temp_file, 'JPEG', quality=95)  # 保留较高质量

            return temp_file
        except Exception as e:
            print(f"❌ 转换JPG失败: {e}")
            raise  # 抛出错误，让上层处理

    def _compress_image(self, source_path):
        """压缩图片至约0.5MB（动态调整质量）"""
        try:
            # 创建临时文件
            temp_file = tempfile.mktemp(suffix='.jpg')

            # 打开图片，动态调整质量
            with Image.open(source_path) as img:
                quality = 80  # 初始质量

                # 循环调整质量，直到文件大小≤0.5MB或质量过低
                while True:
                    img.save(temp_file, 'JPEG', quality=quality, optimize=True)
                    current_size = os.path.getsize(temp_file) / (1024 * 1024)

                    # 退出条件：大小达标或质量过低
                    if current_size <= 0.5 or quality <= 10:
                        break

                    # 根据当前大小调整质量（阶梯式降低）
                    if current_size > 0.7:
                        quality -= 20
                    elif current_size > 0.6:
                        quality -= 10
                    else:
                        quality -= 5

            return temp_file
        except Exception as e:
            print(f"❌ 压缩失败: {e}")
            raise  # 抛出错误，让上层处理


# 测试代码（单独运行时执行）
if __name__ == "__main__":
    # 测试路径（实际使用时由main.py传入）
    test_dir = "./lycnb"
    if not os.path.exists(test_dir):
        os.makedirs(test_dir)
        print(f"创建测试文件夹: {test_dir}")

    # 实例化并测试
    processor = ImageProcessor(test_dir)
    processor.process_images()
    print("测试处理完成！")