import os
import time
import logging
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.common.by import By
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import (TimeoutException, WebDriverException, StaleElementReferenceException)

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    filename='deepseek_text_chat.log'
)
logger = logging.getLogger(__name__)


class DeepseekTextChat:
    """Deepseek文字对话核心功能类"""

    def __init__(self, driver_path="chromedriver.exe"):
        self.driver_path = driver_path  # ChromeDriver路径
        self.driver = None

    def start(self):
        """启动浏览器并打开Deepseek页面，等待用户登录"""
        try:
            logger.info(f"使用ChromeDriver路径: {self.driver_path}")

            # 检查ChromeDriver是否存在
            if not os.path.exists(self.driver_path):
                raise FileNotFoundError(f"ChromeDriver文件不存在: {self.driver_path}")

            # 启动浏览器
            options = webdriver.ChromeOptions()
            options.add_argument("--start-maximized")  # 最大化窗口，避免元素被遮挡
            options.add_argument("--disable-notifications")

            # 添加便携Chrome路径
            chrome_path = self.get_portable_chrome_path()
            if chrome_path:
                options.binary_location = chrome_path
                logger.info(f"使用便携Chrome路径: {chrome_path}")
            else:
                logger.info("未找到有效便携Chrome路径，使用系统默认Chrome")

            self.driver = webdriver.Chrome(
                service=Service(self.driver_path),
                options=options
            )
            self.driver.get("https://chat.deepseek.com/sign_in")
            logger.info("Deepseek页面已打开")

            # 提示用户登录
            print("请在浏览器中完成Deepseek登录操作（登录后页面会显示聊天界面）")
            print("登录完成后请按回车键继续...")
            input()  # 等待用户按回车键确认登录完成

            # 验证登录成功（检测聊天输入框）
            WebDriverWait(self.driver, 30).until(
                EC.presence_of_element_located((By.XPATH, '//*[@id="chat-input"]'))
            )
            print("检测到登录成功，已准备好进行文字对话")

        except WebDriverException as e:
            logger.error(f"浏览器启动失败: {e}")
            raise
        except Exception as e:
            logger.error(f"初始化失败: {e}")
            raise

    def get_portable_chrome_path(self):
        """获取便携Chrome路径（适配当前目录结构）"""
        # 拼接便携Chrome可执行文件路径：当前目录/GoogleChromePortable/App/Chrome-bin/chrome.exe
        chrome_exe_path = os.path.join(
            os.getcwd(),  # 当前工作目录（即"一键完成工具"文件夹）
            'GoogleChromePortable',
            'App',
            'Chrome-bin',
            'chrome.exe'
        )
        # 检查路径是否存在
        if os.path.exists(chrome_exe_path):
            return chrome_exe_path
        else:
            logger.warning(f"便携Chrome路径不存在: {chrome_exe_path}")
            return None  # 路径不存在时返回None，降级使用系统Chrome

    def send_message(self, message, timeout=60):
        """文字对话核心功能：发送消息并获取回复"""
        if not self.driver:
            raise Exception("请先调用start()方法启动浏览器")

        try:
            print(f"\n执行文字对话：{message}")
            logger.info(f"发送文字消息: {message}")

            # 定位输入框并发送消息
            input_box = WebDriverWait(self.driver, 10).until(
                EC.element_to_be_clickable((By.XPATH, '//*[@id="chat-input"]'))
            )
            input_box.clear()
            input_box.send_keys(message)

            # 使用Enter键替代点击发送按钮
            input_box.send_keys(Keys.RETURN)
            print("文字已发送，等待回复...")

            # 等待最新回复元素出现
            reply_xpath = self._get_latest_reply_xpath(timeout)

            # 等待回复内容稳定（处理动态加载和元素刷新）
            print("检测回复内容加载状态...")
            start_time = time.time()
            previous_text = ""
            stable_count = 0

            while time.time() - start_time < timeout:
                try:
                    # 每次都重新获取元素，避免stale element错误
                    reply_elements = self.driver.find_elements(By.XPATH, reply_xpath)
                    current_text = "\n".join([elem.text.strip() for elem in reply_elements if elem.text.strip()])

                    # 检查内容是否稳定
                    if current_text == previous_text and current_text:
                        stable_count += 1
                        if stable_count >= 3:  # 连续3秒内容不变
                            break
                    else:
                        stable_count = 0
                        previous_text = current_text

                    time.sleep(1)

                except StaleElementReferenceException:
                    # 元素已刷新，重置计数器
                    stable_count = 0
                    previous_text = ""
                    time.sleep(1)
                except Exception as e:
                    print(f"临时错误: {e}")
                    time.sleep(1)

            # 最终获取一次完整回复
            reply_elements = self.driver.find_elements(By.XPATH, reply_xpath)
            reply = "\n".join([elem.text.strip() for elem in reply_elements if elem.text.strip()])

            print(f"Deepseek回复：{reply}")
            logger.info(f"收到回复: {reply[:50]}...")
            return reply

        except TimeoutException:
            error_msg = "等待回复超时"
            print(f"文字对话出错：{error_msg}")
            logger.error(error_msg)
            return None
        except Exception as e:
            print(f"文字对话出错：{e}")
            logger.error(f"发送/获取消息失败: {e}")
            return None

    def _get_latest_reply_xpath(self, timeout):
        """获取最新回复的XPath（根据规律生成）"""
        # 分析规律：
        # 第1条回复：div[1]/div[2]/div[2]/p
        # 第2条回复：div[1]/div[4]/div[2]/p[1]
        # 第3条回复：div[1]/div[6]/div[2]/p[1]
        # 规律为：div[1]/div[2n]/div[2]/p[1] （n为回复序号）
        # 先获取回复总数
        container_xpath = '//*[@id="root"]/div/div/div[2]/div[3]/div/div[2]/div/div/div[1]'
        container = WebDriverWait(self.driver, timeout).until(
            EC.presence_of_element_located((By.XPATH, container_xpath))
        )

        # 计算回复数量（每个回复包含在div[2n]中）
        reply_divs = container.find_elements(By.XPATH, './div[position() mod 2 = 0]')  # 取偶数位置的div
        latest_reply_index = len(reply_divs) * 2  # 最新回复的div索引

        if latest_reply_index < 2:
            latest_reply_index = 2  # 至少从第2个div开始

        # 构建最新回复的XPath
        return f'//*[@id="root"]/div/div/div[2]/div[3]/div/div[2]/div/div/div[1]/div[{latest_reply_index}]/div[2]/p'

    def close(self):
        """关闭浏览器"""
        if self.driver:
            try:
                self.driver.quit()
                logger.info("浏览器已关闭")
            except Exception as e:
                logger.error(f"关闭浏览器失败: {e}")

    def run_manual_test(self):
        """运行手动测试模式"""
        print("\n===== 手动测试模式 =====")
        print("输入测试指令或消息进行测试：")
        print("  - 输入消息：发送消息并获取回复")
        print("  - /clear：清除聊天历史")
        print("  - /exit：退出测试模式")

        while True:
            command = input("\n输入指令或消息：").strip()

            if command.lower() == "/exit":
                print("退出手动测试模式")
                break
            elif command.lower() == "/clear":
                # 尝试清除聊天历史（Deepseek的清空按钮可能需要调整XPath）
                try:
                    # 尝试点击更多选项按钮
                    more_btn = WebDriverWait(self.driver, 10).until(
                        EC.element_to_be_clickable((By.XPATH, '//button[contains(@class, "more-actions")]'))
                    )
                    more_btn.click()

                    # 尝试点击清空对话按钮
                    clear_btn = WebDriverWait(self.driver, 10).until(
                        EC.element_to_be_clickable((By.XPATH, '//span[contains(text(), "清空对话")]'))
                    )
                    clear_btn.click()

                    # 确认弹窗（如果有）
                    try:
                        confirm_btn = WebDriverWait(self.driver, 5).until(
                            EC.element_to_be_clickable((By.XPATH, '//button[contains(text(), "确认")]'))
                        )
                        confirm_btn.click()
                        print("聊天历史已清除")
                    except TimeoutException:
                        print("未检测到确认弹窗，可能已清除")
                except Exception as e:
                    print(f"清除聊天历史失败: {e}")
            else:
                # 发送普通消息
                self.send_message(command)


# 主程序入口
if __name__ == "__main__":
    driver_path = "chromedriver.exe"  # ChromeDriver路径（与脚本同目录）

    # 检查ChromeDriver是否存在
    if not os.path.exists(driver_path):
        print(f"错误: ChromeDriver不存在于当前目录 - {driver_path}")
        print("请下载匹配版本的ChromeDriver并放在与脚本相同的目录下")
        print("下载地址: https://sites.google.com/chromium.org/driver/")
    else:
        try:
            # 初始化并启动
            deepseek = DeepseekTextChat(driver_path)
            deepseek.start()

            # 运行手动测试
            deepseek.run_manual_test()

            print("\n测试完成，按任意键退出...")
            input()
        except Exception as e:
            print(f"程序运行出错: {e}")
        finally:
            # 确保浏览器关闭
            if 'deepseek' in locals() and deepseek.driver:
                deepseek.close()