import os
import time
import logging
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import (TimeoutException, WebDriverException, StaleElementReferenceException)

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    filename='doubao_text_chat.log'
)
logger = logging.getLogger(__name__)


class DoubaoTextChat:
    """仅保留文字对话核心功能（lyc666）"""

    def __init__(self, driver_path="chromedriver.exe"):
        self.driver_path = driver_path  # ChromeDriver路径
        self.driver = None

    def start(self):
        """启动浏览器并打开豆包页面，等待用户登录"""
        try:
            logger.info(f"使用ChromeDriver路径: {self.driver_path}")

            # 检查ChromeDriver是否存在
            if not os.path.exists(self.driver_path):
                raise FileNotFoundError(f"ChromeDriver文件不存在: {self.driver_path}")

            # 启动浏览器
            options = webdriver.ChromeOptions()
            options.add_argument("--start-maximized")  # 最大化窗口，避免元素被遮挡
            options.add_argument("--disable-notifications")
            self.driver = webdriver.Chrome(
                service=Service(self.driver_path),
                options=options
            )
            self.driver.get("https://doubao.com")
            logger.info("豆包页面已打开")

            # 提示用户登录
            print("请在浏览器中完成豆包登录操作并关闭深度思考模式（登录后页面会显示聊天界面）")
            # 等待登录成功（检测聊天输入框）
            WebDriverWait(self.driver, 60).until(
                EC.presence_of_element_located((By.CSS_SELECTOR, 'textarea[data-testid="chat_input_input"]'))
            )
            print("检测到登录成功，已准备好进行文字对话")

        except WebDriverException as e:
            logger.error(f"浏览器启动失败: {e}")
            raise
        except Exception as e:
            logger.error(f"初始化失败: {e}")
            raise

    def lyc666(self, message, timeout=60):
        """文字对话核心功能：发送消息并获取回复"""
        if not self.driver:
            raise Exception("请先调用start()方法启动浏览器")

        try:
            print(f"\n执行文字对话：{message}")
            logger.info(f"发送文字消息: {message}")

            # 定位输入框并发送消息
            input_box = WebDriverWait(self.driver, 10).until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, 'textarea[data-testid="chat_input_input"]'))
            )
            input_box.clear()
            input_box.send_keys(message)

            # 定位发送按钮并点击
            send_btn = WebDriverWait(self.driver, 10).until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, 'button[data-testid="chat_input_send_button"]'))
            )
            send_btn.click()
            print("文字已发送，等待回复...")

            # 等待最新回复元素出现
            # 使用动态XPath来获取最新的回复
            reply_xpath = self._get_latest_reply_xpath(timeout)

            # 等待回复内容稳定（处理动态加载和元素刷新）
            print("检测回复内容加载状态...")
            start_time = time.time()
            previous_text = ""
            stable_count = 0

            while time.time() - start_time < timeout:
                try:
                    # 每次都重新获取元素，避免stale element错误
                    reply_element = self.driver.find_element(By.XPATH, reply_xpath)
                    current_text = reply_element.text.strip()

                    # 检查内容是否稳定
                    if current_text == previous_text and current_text:
                        stable_count += 1
                        if stable_count >= 3:  # 连续3秒内容不变
                            break
                    else:
                        stable_count = 0
                        previous_text = current_text

                    time.sleep(1)

                except StaleElementReferenceException:
                    # 元素已刷新，重置计数器
                    stable_count = 0
                    previous_text = ""
                    time.sleep(1)
                except Exception as e:
                    print(f"临时错误: {e}")
                    time.sleep(1)

            # 最终获取一次完整回复
            reply_element = self.driver.find_element(By.XPATH, reply_xpath)
            reply = reply_element.text.strip()

            print(f"豆包回复：{reply}")
            logger.info(f"收到回复: {reply[:50]}...")
            return reply

        except TimeoutException:
            error_msg = "等待回复超时"
            print(f"文字对话出错：{error_msg}")
            logger.error(error_msg)
            return None
        except Exception as e:
            print(f"文字对话出错：{e}")
            logger.error(f"发送/获取消息失败: {e}")
            return None

    def _get_latest_reply_xpath(self, timeout):
        """获取最新回复的XPath"""
        # 先找到所有可能的回复容器
        container_xpath = '//*[@id="root"]/div[1]/div/div[3]/div/main/div/div/div[2]/div/div[1]/div/div/div[2]'
        container = WebDriverWait(self.driver, timeout).until(
            EC.presence_of_element_located((By.XPATH, container_xpath))
        )

        # 获取所有回复元素
        reply_containers = container.find_elements(By.XPATH, './*')

        # 计算最新回复的索引
        # 索引通常是2,4,6这样的偶数，对应div[2],div[4],div[6]
        reply_count = len(reply_containers)
        latest_index = reply_count if reply_count % 2 == 0 else reply_count - 1

        if latest_index < 2:
            latest_index = 2  # 默认从2开始

        # 构建最新回复的XPath
        return f'//*[@id="root"]/div[1]/div/div[3]/div/main/div/div/div[2]/div/div[1]/div/div/div[2]/div[{latest_index}]/div/div/div/div/div/div/div[1]/div/div/div'

    def close(self):
        """关闭浏览器"""
        if self.driver:
            try:
                self.driver.quit()
                logger.info("浏览器已关闭")
            except Exception as e:
                logger.error(f"关闭浏览器失败: {e}")

    def run_manual_test(self):
        """运行手动测试模式"""
        print("\n===== 手动测试模式 =====")
        print("输入测试指令或消息进行测试：")
        print("  - 输入消息：发送消息并获取回复")
        print("  - /clear：清除聊天历史")
        print("  - /exit：退出测试模式")

        while True:
            command = input("\n输入指令或消息：").strip()

            if command.lower() == "/exit":
                print("退出手动测试模式")
                break
            elif command.lower() == "/clear":
                # 尝试清除聊天历史
                try:
                    clear_btn = WebDriverWait(self.driver, 10).until(
                        EC.element_to_be_clickable((By.XPATH, '//button[contains(text(), "清空对话")]'))
                    )
                    clear_btn.click()

                    # 确认弹窗（如果有）
                    try:
                        confirm_btn = WebDriverWait(self.driver, 5).until(
                            EC.element_to_be_clickable((By.XPATH, '//button[contains(text(), "确认")]'))
                        )
                        confirm_btn.click()
                        print("聊天历史已清除")
                    except TimeoutException:
                        print("未检测到确认弹窗，可能已清除")
                except Exception as e:
                    print(f"清除聊天历史失败: {e}")
            else:
                # 发送普通消息
                self.lyc666(command)


# 主程序入口
if __name__ == "__main__":
    driver_path = "chromedriver.exe"  # ChromeDriver路径（与脚本同目录）

    # 检查ChromeDriver是否存在
    if not os.path.exists(driver_path):
        print(f"错误: ChromeDriver不存在于当前目录 - {driver_path}")
        print("请下载匹配版本的ChromeDriver并放在与脚本相同的目录下")
        print("下载地址: https://sites.google.com/chromium.org/driver/")
    else:
        try:
            # 初始化并启动
            doubao = DoubaoTextChat(driver_path)
            doubao.start()

            # 运行手动测试
            doubao.run_manual_test()

            print("\n测试完成，按任意键退出...")
            input()

        except Exception as e:
            print(f"程序运行出错: {e}")
        finally:
            # 确保浏览器关闭
            if 'doubao' in locals() and doubao.driver:
                doubao.close()